<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\Carteira;

class InstallClubGrow extends Command
{
    /**
     * O nome e a assinatura do comando no console.
     * (É assim que o chamamos: php artisan clubgrow:install)
     */
    protected $signature = 'clubgrow:install';

    /**
     * A descrição do comando.
     */
    protected $description = 'Executa o instalador da plataforma ClubGrow (migrações, admin, etc.)';

    /**
     * Executa a lógica do comando.
     */
    public function handle()
    {
        $this->info('======================================');
        $this->info('🚀 INICIANDO O INSTALADOR DO CLUBGROW 🚀');
        $this->info('======================================');

        // 1. Gerar a Chave da Aplicação (se não existir no .env)
        if (!config('app.key')) {
            $this->line('1. Gerando chave da aplicação...');
            Artisan::call('key:generate');
            $this->info('Chave gerada com sucesso.');
        } else {
            $this->line('1. Chave da aplicação já existe. Pulando.');
        }

        // 2. Executar as Migrações do Banco de Dados
        $this->line('2. Executando migrações do banco de dados...');
        try {
            // Usamos '--force' para rodar em produção sem perguntar
            Artisan::call('migrate', ['--force' => true]);
            $this->info(Artisan::output()); // Mostra a saída do comando migrate
            $this->info('Migrações concluídas com sucesso.');
        } catch (\Exception $e) {
            $this->error('ERRO AO EXECUTAR MIGRAÇÕES:');
            $this->error($e->getMessage());
            return 1; // Termina o comando com erro
        }

        // 3. Criar o Utilizador Administrador
        $this->line('3. Criando o utilizador Administrador...');
        if (User::where('tipo_usuario', 'admin')->count() > 0) {
            $this->warn('Um utilizador Admin já existe. Pulando criação do admin.');
        } else {
            $this->info('Por favor, forneça os dados para a conta de Administrador:');
            
            $nome = $this->ask('Nome completo do Admin:');
            $email = $this->ask('E-mail do Admin:');
            $senha = $this->secret('Senha do Admin (mín. 8 caracteres):');

            // Validação
            if (strlen($senha) < 8 || !$nome || !$email) {
                $this->error('Dados inválidos. A senha deve ter no mínimo 8 caracteres. Instalação falhou.');
                return 1;
            }

            // Criação
            try {
                $admin = User::create([
                    'nome_completo' => $nome,
                    'email' => $email,
                    'senha' => $senha, // O Model User já faz o Hash automático
                    'tipo_usuario' => 'admin',
                    'status' => 'ativo', // Admin já começa ativo
                    'email_verificado_em' => now(), // Admin já começa verificado
                ]);

                // Cria a carteira para o Admin
                Carteira::create(['usuario_id' => $admin->id]);

                $this->info('Utilizador Administrador criado com sucesso!');

            } catch (\Exception $e) {
                $this->error('ERRO AO CRIAR ADMIN:');
                $this->error($e->getMessage());
                return 1;
            }
        }

        // 4. Limpar o Cache
        $this->line('4. Limpando o cache de configuração e rotas...');
        Artisan::call('optimize:clear');
        $this->info('Cache limpo.');

        $this->info('========================================');
        $this->info('✅ Instalação do ClubGrow concluída!');
        $this->info('========================================');
        return 0; // Termina com sucesso
    }
}