<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\PerfilKyc;
use App\Models\Transacao;
use App\Models\TicketSuporte;
use App\Models\InvestimentoUsuario;

class DashboardController extends Controller
{
    /**
     * Busca as estatísticas (widgets) para o painel principal do Admin.
     * (GET /api/admin/dashboard)
     */
    public function getDashboardStats(Request $request)
    {
        // --- Widgets da sua Lista (Contagem de Utilizadores) ---
        // Filtramos por 'investidor' para não contar outros admins
        
        $usuariosBanidos = User::where('tipo_usuario', 'investidor')
                                ->where('status', 'inativo')
                                ->count();
        
        $emailNaoVerificado = User::where('tipo_usuario', 'investidor')
                                  ->whereNull('email_verificado_em')
                                  ->count();
        
        $celularNaoVerificado = User::where('tipo_usuario', 'investidor')
                                    ->whereNull('telefone_verificado_em')
                                    ->count();

        // --- Widgets de KYC ---
        
        $kycPendente = PerfilKyc::where('status_kyc', 'pendente')->count();

        // Utilizadores que NUNCA enviaram um KYC
        $kycNaoVerificado = User::where('tipo_usuario', 'investidor')
                                ->whereDoesntHave('kyc')
                                ->count();

        // --- Widgets Proativos (Filas de Trabalho) ---
        
        $depositosPendentes = Transacao::where('tipo_transacao', 'deposito')
                                        ->where('status', 'pendente')
                                        ->count();
        
        $retiradasPendentes = Transacao::where('tipo_transacao', 'retirada')
                                        ->where('status', 'pendente')
                                        ->count();
        
        $ticketsAbertos = TicketSuporte::where('status', 'aberto')->count();

        // --- Widgets Proativos (Financeiro Rápido) ---
        
        // Total de dinheiro que está investido ativamente
        $totalInvestidoAtivo = InvestimentoUsuario::where('status', 'ativo')->sum('valor_investido');
        
        // Total de lucro da plataforma (ainda não temos taxas,
        // então vamos mostrar o total depositado este mês)
        $totalDepositadoMes = Transacao::where('tipo_transacao', 'deposito')
                                        ->where('status', 'concluido')
                                        ->whereMonth('created_at', now()->month)
                                        ->whereYear('created_at', now()->year)
                                        ->sum('valor');

        // --- Monta a Resposta ---
        return response()->json([
            // Widgets baseados na sua lista (para os contadores)
            'widgets_contagem' => [
                'usuarios_banidos' => $usuariosBanidos,
                'email_nao_verificado' => $emailNaoVerificado,
                'celular_nao_verificado' => $celularNaoVerificado,
                'kyc_nao_verificado' => $kycNaoVerificado,
            ],
            // Widgets de Ação (filas de trabalho, com link)
            'widgets_acao' => [
                'kyc_pendente' => $kycPendente,
                'depositos_pendentes' => $depositosPendentes,
                'retiradas_pendentes' => $retiradasPendentes,
                'tickets_abertos' => $ticketsAbertos,
            ],
            // Widgets Financeiros (gráficos)
            'widgets_financeiro' => [
                'total_investido_ativo' => (float) $totalInvestidoAtivo,
                'total_depositado_mes' => (float) $totalDepositadoMes,
            ]
        ], 200);
    }
}