<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TicketSuporte;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SuporteController extends Controller
{
    /**
     * LISTAR todos os tickets para o Admin
     * (GET /api/admin/suporte/tickets)
     */
    public function index(Request $request)
    {
        // Começa a query, puxando o nome do utilizador de cada ticket
        $query = TicketSuporte::with('user:id,nome_completo');

        // Lógica de Filtro Proativa:
        // Por defeito, o admin não quer ver tickets "fechados".
        // Ele quer ver a fila de trabalho ("aberto" ou "respondido").
        if ($request->has('status')) {
            $query->where('status', $request->status);
        } else {
            $query->whereIn('status', ['aberto', 'respondido']);
        }
        
        // Ordena pelos que precisam de resposta mais urgente (os mais antigos)
        // ou pela última resposta
        $tickets = $query->orderBy('ultima_resposta_em', 'desc')
                         ->paginate(20);

        return response()->json($tickets, 200);
    }

    /**
     * VER um ticket específico e a sua conversa
     * (GET /api/admin/suporte/tickets/{id})
     */
    public function show(Request $request, $id)
    {
        // Busca o ticket e carrega:
        // 1. O 'user' (cliente) que abriu o ticket
        // 2. As 'mensagens' (conversa)
        // 3. O 'user' (autor) de cada mensagem (para saber quem disse o quê)
        $ticket = TicketSuporte::with([
                                'user:id,nome_completo,email', 
                                'mensagens.user:id,nome_completo'
                            ])
                            ->find($id);

        if (!$ticket) {
            return response()->json(['message' => 'Ticket não encontrado.'], 404);
        }

        return response()->json($ticket, 200);
    }

    /**
     * RESPONDER a um ticket (Ação do Admin)
     * (POST /api/admin/suporte/tickets/{id}/responder)
     */
    public function responder(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'mensagem' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $admin = Auth::user(); // Pega o Admin logado
        $ticket = TicketSuporte::find($id);

        if (!$ticket) {
            return response()->json(['message' => 'Ticket não encontrado.'], 404);
        }

        if ($ticket->status === 'fechado') {
            return response()->json(['message' => 'Este ticket está fechado.'], 403);
        }

        // 1. Cria a nova mensagem
        $mensagem = $ticket->mensagens()->create([
            'usuario_id' => $admin->id, // A mensagem é do ADMIN
            'mensagem' => $request->mensagem,
            // 'anexo_path' => null // Lógica de upload do admin viria aqui
        ]);

        // 2. Atualiza o status do ticket
        // Quando o Admin responde, o status muda para "respondido"
        $ticket->status = 'respondido';
        $ticket->ultima_resposta_em = now();
        $ticket->save();

        return response()->json([
            'message' => 'Resposta enviada!',
            'mensagem' => $mensagem->load('user:id,nome_completo') // Retorna a mensagem com o nome do admin
        ], 201);
    }

    /**
     * FECHAR um ticket
     * (POST /api/admin/suporte/tickets/{id}/fechar)
     */
    public function fechar(Request $request, $id)
    {
        $ticket = TicketSuporte::find($id);

        if (!$ticket) {
            return response()->json(['message' => 'Ticket não encontrado.'], 404);
        }

        $ticket->status = 'fechado';
        $ticket->save();

        return response()->json(['message' => 'Ticket fechado com sucesso.'], 200);
    }
}