<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Auth;    
use Illuminate\Support\Facades\Validator; 

class UsuarioController extends Controller
{
    /**
     * Função LISTAR UTILIZADORES (GET /api/admin/usuarios)
     * Lista todos os utilizadores com filtros.
     */
    public function listarUsuarios(Request $request)
    {
        // Começa a query, carregando as relações (eager loading)
        $query = User::with(['carteira', 'kyc'])
                     // Não mostrar outros admins na lista de gestão
                     ->where('tipo_usuario', '!=', 'admin');

        // --- Filtros (baseados no seu menu lateral) ---

        // ?status=ativo ou ?status=inativo
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // ?kyc_status=pendente ou ?kyc_status=rejeitado
        if ($request->has('kyc_status')) {
            // whereHas filtra utilizadores que TÊM um kyc com o status...
            $query->whereHas('kyc', function ($q) use ($request) {
                $q->where('status_kyc', $request->kyc_status);
            });
        }
        
        // ?kyc_status=nao_verificado (utilizadores que NUNCA enviaram KYC)
        if ($request->query('kyc_status') === 'nao_verificado') {
            // whereDoesntHave filtra utilizadores que NÃO TÊM um registo de kyc
            $query->whereDoesntHave('kyc');
        }

        // ?email_verificado=false
        if ($request->query('email_verificado') === 'false') {
            $query->whereNull('email_verificado_em');
        }
        
        // ?telefone_verificado=false
        if ($request->query('telefone_verificado') === 'false') {
            $query->whereNull('telefone_verificado_em');
        }
        
        // ?com_equilibrio=true
        if ($request->query('com_equilibrio') === 'true') {
             $query->whereHas('carteira', function ($q) {
                $q->where('saldo_disponivel', '>', 0);
            });
        }

        // --- Fim dos Filtros ---

        // Ordena e pagina o resultado
        $usuarios = $query->orderBy('created_at', 'desc')->paginate(25);

        return response()->json($usuarios, 200);
    }
    
	
	
    // TODO: Adicionar função para ver detalhes de um utilizador (getUsuario)
	
	// ... (depois da função listarUsuarios)

/**
 * Função GET DETALHES (GET /api/admin/usuarios/{id})
 * Busca os detalhes completos de um usuário para o admin revisar.
 */
public function getUsuarioDetalhes(Request $request, $id)
{
    // Busca o usuário e carrega todas as suas relações
    $user = User::with([
                'carteira', 
                'kyc.revisadoPor:id,nome_completo', // Pega o KYC e o nome do admin que revisou
                'investimentos', 
                'transacoes'
            ])
            ->find($id);

    if (!$user) {
        return response()->json(['message' => 'Usuário não encontrado.'], 404);
    }

    // Segurança: Não permitir que um admin veja os dados de outro admin
    if ($user->tipo_usuario === 'admin') {
         return response()->json(['message' => 'Acesso não autorizado.'], 403);
    }

    return response()->json($user, 200);
}

/**
 * Função GERIR KYC (POST /api/admin/usuarios/{id}/manage-kyc)
 * Aprova ou Rejeita o KYC de um usuário.
 */
public function manageKyc(Request $request, $id)
{
    // 1. Validar o input do Admin
    $validator = Validator::make($request->all(), [
        'status_kyc' => 'required|in:aprovado,rejeitado', // O status só pode ser um destes
        // Se for 'rejeitado', o campo 'notas_rejeicao' é obrigatório
        'notas_rejeicao' => 'required_if:status_kyc,rejeitado|string|nullable',
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    // 2. Pegar os participantes
    $admin = Auth::user(); // O Admin que está a fazer a ação
    $targetUser = User::find($id); // O Usuário que está a ser gerido

    if (!$targetUser) {
        return response()->json(['message' => 'Usuário alvo não encontrado.'], 404);
    }

    // 3. Encontrar o perfil KYC
    // Usamos firstOrCreate para o caso (raro) de o admin querer rejeitar
    // um utilizador que nem sequer tem um registo de KYC.
    $kycProfile = $targetUser->kyc()->firstOrCreate(['usuario_id' => $targetUser->id]);

    // 4. Atualizar o Perfil KYC
    $kycProfile->status_kyc = $request->status_kyc;
    $kycProfile->data_revisao = now();
    $kycProfile->revisado_por_admin_id = $admin->id;
    $kycProfile->notas_rejeicao = $request->notas_rejeicao;
    $kycProfile->save();

    // 5. ATUALIZAR O STATUS PRINCIPAL DO USUÁRIO (MUITO IMPORTANTE)
    if ($request->status_kyc === 'aprovado') {
        // Se o KYC foi aprovado, ativamos a conta principal do usuário
        // (Isto permite que ele faça login, como definido no AuthController)
        $targetUser->status = 'ativo';
    } else {
        // Se o KYC foi rejeitado, desativamos a conta
        $targetUser->status = 'inativo';
    }
    $targetUser->save();

    // 6. Retornar a resposta
    return response()->json([
        'message' => 'Status do KYC atualizado com sucesso!',
        'usuario' => $targetUser->load('kyc'), // Retorna o usuário atualizado
    ], 200);
}
	
	
	
    // TODO: Adicionar função para aprovar/rejeitar KYC (manageKyc)
}