<?php

namespace App\Http\Controllers\Api;

use App\Models\Carteira;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User; // Nosso Model User
use Illuminate\Support\Facades\Auth; // Facade de Autenticação
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;

class AuthController extends Controller
{
    /**
     * REGISTRAR NOVO USUÁRIO
     */
    public function registrar(Request $request)
    {
        // 1. Validação
        $validator = Validator::make($request->all(), [
            'nome_completo' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:usuarios',
            'senha' => ['required', 'confirmed', Password::min(8)->mixedCase()->numbers()],
            'codigo_indicacao_pai' => 'nullable|string|exists:usuarios,codigo_indicacao', // Opcional
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // 2. Achar o ID do indicador (se existir)
        $indicadorId = null;
        if ($request->codigo_indicacao_pai) {
            $indicador = User::where('codigo_indicacao', $request->codigo_indicacao_pai)->first();
            if ($indicador) {
                $indicadorId = $indicador->id;
            }
        }

        // 3. Criar o Usuário
        $user = User::create([
            'nome_completo' => $request->nome_completo,
            'email' => $request->email,
            'senha' => $request->senha, // O Model User vai criptografar isso!
            'status' => 'ativo', // Definido como 'ativo' para testes. Mude para 'pendente' se precisar de verificação de e-mail.
            'tipo_usuario' => 'investidor',
            'indicado_por_usuario_id' => $indicadorId,
        ]);
	
		// *** LINHA ADICIONADA ***
		// Cria uma carteira (saldo) vazia para o novo usuário
		Carteira::create(['usuario_id' => $user->id]);
		

        // 4. Gerar o Token de API (Sanctum)
        $token = $user->createToken('auth_token_clubgrow')->plainTextToken;

        // 5. Resposta
        return response()->json([
            'message' => 'Usuário registrado com sucesso!',
            'usuario' => $user,
            'token_acesso' => $token,
        ], 201);
    }

    /**
     * FAZER LOGIN
     */
    public function login(Request $request)
    {
        // 1. Validação
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'senha' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // 2. Tentar autenticar
        if (!Auth::attempt($request->only('email', 'senha'))) {
            return response()->json(['message' => 'Credenciais inválidas.'], 401);
        }

        // 3. Pegar o usuário
        $user = User::where('email', $request->email)->firstOrFail();

        // 4. Checar Status (MUITO IMPORTANTE)
        if ($user->status !== 'ativo') {
             return response()->json([
                'message' => 'Esta conta está inativa ou pendente. Contate o suporte.'
            ], 403);
        }

        // 5. Gerar o Token
        $token = $user->createToken('auth_token_clubgrow')->plainTextToken;

        // 6. Resposta
        return response()->json([
            'message' => 'Login bem-sucedido!',
            'usuario' => $user,
            'token_acesso' => $token,
        ], 200);
    }

    /**
     * FAZER LOGOUT
     * (Esta rota será protegida, só funciona se o usuário estiver logado)
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return response()->json(['message' => 'Logout realizado com sucesso.'], 200);
    }

    /**
     * BUSCAR PERFIL DO USUÁRIO LOGADO
     * (Esta rota também será protegida)
     */
    public function meuPerfil(Request $request)
    {
        return response()->json($request->user(), 200);
    }
}