<?php

namespace App\Http\Controllers\Api\Cliente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth; // Usaremos o Auth para pegar o usuário logado
use Illuminate.Support\Facades\Storage; 
use Illuminate\Support\Facades\Validator; 
use App\Models\PerfilKyc; 

class PerfilController extends Controller
{
    /**
     * Busca os dados (widgets) para o painel do cliente.
     */
    public function getDashboardStats(Request $request)
    {
        // Pega o usuário que está logado (autenticado via Sanctum)
        $user = Auth::user();

        // 1. Busca a carteira (Graças à relação que criámos no User.php)
        // Usamos 'firstOrCreate' como uma segurança caso a carteira não tenha sido criada no registo
        $carteira = $user->carteira()->firstOrCreate(['usuario_id' => $user->id]);

        // 2. Pega os saldos
        $equilibrio = $carteira->saldo_disponivel;
        $investimento_total = $carteira->saldo_bloqueado; // Total que está investido

        // 3. Pega os totais das Transações (Graças à relação 'transacoes')
        $depositos = $user->transacoes()
                          ->where('tipo_transacao', 'deposito')
                          ->where('status', 'concluido')
                          ->sum('valor');
        
        $retiradas = $user->transacoes()
                          ->where('tipo_transacao', 'retirada')
                          ->where('status', 'concluido')
                          ->sum('valor');
        
        $lucro_total = $user->transacoes()
                            ->where('tipo_transacao', 'retorno_lucro')
                            ->where('status', 'concluido')
                            ->sum('valor');

        // 4. Pega o total de investimentos (Graças à relação 'investimentos')
        $casos = $user->investimentos()->where('status', 'ativo')->count();

        // 5. Retorna tudo num formato JSON limpo
        return response()->json([
            'equilibrio' => (float) $equilibrio,
            'deposito' => (float) $depositos,
            'retirar' (retiradas) => (float) $retiradas,
            'casos' => (int) $casos,
            'investimento_total' => (float) $investimento_total,
            'lucro_total' => (float) $lucro_total,
        ], 200);
    }


/**
 * Função SUBMETER KYC (POST /api/meu-perfil/kyc)
 * Recebe os documentos do utilizador e guarda-os para aprovação.
 */
public function submitKyc(Request $request)
{
    // 1. Validar os dados recebidos
    // O frontend (React) deve enviar os ficheiros
    $validator = Validator::make($request->all(), [
        'documento_tipo' => 'required|string|in:RG,CNH,Passaporte',
        'doc_frente' => 'required|file|image|mimes:jpg,jpeg,png|max:2048', // Imagem, máx 2MB
        'doc_verso' => 'required|file|image|mimes:jpg,jpeg,png|max:2048',
        'selfie' => 'required|file|image|mimes:jpg,jpeg,png|max:2048',
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    $user = Auth::user();

    // 2. Encontrar ou Criar o perfil KYC
    $kycProfile = $user->kyc()->firstOrCreate(['usuario_id' => $user->id]);

    // 3. Verificar se já não está aprovado ou pendente
    if ($kycProfile->status_kyc === 'aprovado') {
        return response()->json(['message' => 'O seu KYC já está aprovado.'], 400);
    }
    if ($kycProfile->status_kyc === 'pendente') {
        return response()->json(['message' => 'O seu KYC já está pendente de aprovação.'], 400);
    }

    // 4. --- Processar e Guardar Ficheiros ---
    // Vamos guardar os ficheiros numa pasta única para este utilizador (ex: kyc/user_123)
    $userPath = 'kyc/user_' . $user->id;

    // Apaga ficheiros antigos (se for um reenvio após rejeição)
    Storage::deleteDirectory($userPath); 

    // Guarda os novos ficheiros e obtém o caminho
    $pathFrente = $request->file('doc_frente')->store($userPath, 'public');
    $pathVerso = $request->file('doc_verso')->store($userPath, 'public');
    $pathSelfie = $request->file('selfie')->store($userPath, 'public');

    // 5. Atualizar o registo no banco de dados
    $kycProfile->documento_tipo = $request->documento_tipo;
    $kycProfile->documento_frente_path = $pathFrente;
    $kycProfile->documento_verso_path = $pathVerso;
    $kycProfile->selfie_path = $pathSelfie;
    $kycProfile->status_kyc = 'pendente'; // Define como pendente para o Admin aprovar
    $kycProfile->data_submissao = now();
    $kycProfile->notas_rejeicao = null; // Limpa notas de rejeição antigas
    $kycProfile->save();

    return response()->json([
        'message' => 'Documentos enviados com sucesso! Aguarde a aprovação.',
        'kyc' => $kycProfile
    ], 200);
	
}